;=============================================================================
;	Filename:	bootload 24X1.asm
;=============================================================================
;	Author:		Jim Veatch
;	Callsign:	WA2EUJ
;	Revision:	1.00
;	Date:		30 October 2006
;	Assembled using MPASM V7.20
;=============================================================================
;	Include Files:	p16f871.inc	
;=============================================================================
;	Boot code to receive a hex file containing user code from a
;	serial port and write it to program memory. Tests a pin to see
;	if code should be downloaded. Receives hex file using USART and
;	software handshaking. Does error checking on data and writes to
;	program memory. Waits for reset and then starts user code running.
;=============================================================================

	#include <p16f871.inc>

	__CONFIG _CP_OFF & _WDT_OFF & _BODEN_ON & _PWRTE_ON & _XT_OSC & _WRT_ENABLE_ON & _LVP_OFF & _DEBUG_OFF & _CPD_OFF

;-----------------------------------------------------------------------------
;Constants
RESET_V			EQU	0x0000					; Address of RESET Vector
ISR_V			EQU	0x0004					; Address of Interrupt Vector

TEST_INPUT		EQU	0						;Port B Pin 0 input indicates download

BAUD_9600		equ	0x81					;ASYNC port baud rate 9600
BAUD_19200		equ	0x40					;ASYNC port baud rate 19200

;Fosc is 20MHz
;
;
; LCD Module commands
;
; LCD Commands
LCD_CLEAR		equ	0x01					; Clear the display
LCD_HOME		equ	0x02					; Cursor Home

;Variables in bank0

		CBLOCK	0x20
; Don't reuse in user code
		LCD_TEMP:		1					; LCD subroutines internal use
		TABLE_INDEX:	1					; Index to table strings
		COUNT:			1					; A counter
		DELAY:			1					; Used in DELAYxxx routines
		X_DELAY:		1					; Used in X_DELAYxxx routines
		FP_Switch:		1					; Front Panel Switch State

; OK to Reuse
		Prog_Char:		1					; LCD Progress character storage
		AddressH:		1					; flash program memory address high byte
		AddressL:		1					; flash program memory address low byte
		NumWords:		1					; number of words in line of hex file
		Checksum:		1					; byte to hold checksum of incoming data
		Counter:		1					; to count words being saved or programmed
		TestByte:		1					; byte to show reset vector code received
		HexByte:		1					; byte from 2 incoming ascii characters
		DataPointer:	1					; pointer to data in buffer
		DataArray:	0x40					; buffer for storing incoming data
		ENDC

;-----------------------------------------------------------------------------
;Macros to select the register bank
;Many bank changes can be optimised when only one STATUS bit changes

Bank0			MACRO						;macro to select data RAM bank 0
				bcf		STATUS,RP0
				bcf		STATUS,RP1
				ENDM

Bank1			MACRO						;macro to select data RAM bank 1
				bsf		STATUS,RP0
				bcf		STATUS,RP1
				ENDM

Bank2			MACRO						;macro to select data RAM bank 2
				bcf		STATUS,RP0
				bsf		STATUS,RP1
				ENDM

Bank3			MACRO						;macro to select data RAM bank 3
				bsf		STATUS,RP0
				bsf		STATUS,RP1
				ENDM

;=============================================================================
;Reset vector code
	
				ORG		RESET_V				; RESET vector location
ResetVector:
				CLRF	PORTD
				CLRF	PORTE				; ALL PORT output should output Low.
  				goto    Main				;go to boot loader

;*****************************************************************************
; This is the Periperal Interrupt routine. Should NOT get here
;*****************************************************************************
				ORG		ISR_V				; Interrupt vector location
INTERRUPT		BCF     STATUS, RP0			; Select bank 0
				GOTO	INTERRUPT

;*****************************************************************************
; Main User Code
;*****************************************************************************
				ORG		0x06
START										; POWER_ON Reset (Beginning of user program)

;*****************************************************************************
; Display some stuff
; OK
;*****************************************************************************
				movlw	LCD_HOME			; Position cursor leftmost on first line
				call	LCDPUTCMD
				movlw	'N'
				CALL	LCDPUTCHAR			; Display character
				movlw	'o'
				CALL	LCDPUTCHAR			; Display character
				movlw	' '
				CALL	LCDPUTCHAR			; Display character
				movlw	'U'
				CALL	LCDPUTCHAR			; Display character
				movlw	's'
				CALL	LCDPUTCHAR			; Display character
				movlw	'e'
				CALL	LCDPUTCHAR			; Display character
				movlw	'r'
				CALL	LCDPUTCHAR			; Display character
				movlw	' '
				CALL	LCDPUTCHAR			; Display character
				movlw	'C'
				CALL	LCDPUTCHAR			; Display character
				movlw	'o'
				CALL	LCDPUTCHAR			; Display character
				movlw	'd'
				CALL	LCDPUTCHAR			; Display character
				movlw	'e'
				CALL	LCDPUTCHAR			; Display character


;*****************************************************************************
; Program ends here
; OK
;*****************************************************************************
LOOP:
				GOTO	LOOP				; Stay here forever

;=============================================================================
;Start of boot code in upper memory traps accidental entry into boot code area

		ORG	0x06f0		;Use last part of page0 

StartOfBoot:
TrapError:		goto	TrapError		;trap error and wait for reset

;-----------------------------------------------------------------------------
;Main boot code routine
;Tests to see if a load should occur and if valid user code exists

Main:		
				BSF		STATUS, RP0			; Select bank 1
				MOVLW	0x0 				; RE2-0 outputs
				MOVWF	TRISE
				BCF		STATUS, RP0			; Select bank 0
				MOVLW	.250
				CALL	X_DELAY100			; 250 * 0.1mS = 25mS

				call	FP_READ				; Read and store FP switches
				CALL	LCDINIT				; Initialize LCDisplay;			
;*****************************************************************************
; Check for MENU button pres and enter boot loader if it is...
; OK
;*****************************************************************************
				btfsc	FP_Switch,0			; Look for MENU button press
				goto	START				; If not pressed START user code

				movlw	')'			
				movwf	Prog_Char			; Progress Character

				movlw	LCD_HOME			; Position cursor leftmost on first line
				call	LCDPUTCMD
				movlw	'B'
				CALL	LCDPUTCHAR			; Display character
				movlw	'L'
				CALL	LCDPUTCHAR			; Display character

;-----------------------------------------------------------------------------
;Start of routine to load and program new code

Loader:		
				call	SerialSetup			;set up serial port

			
;-----------------------------------------------------------------------------
;Get new line of hex file starting with ':'
;Get first 8 bytes after ':' and extract address and number of bytes
				movlw	0x11				;send an XON
				call	SerialTransmit		;transmit progress indicator back

GetNewLine:
				call	SerialReceive		;get new byte from serial port
				xorlw	':'					;check if ':' received
				btfss	STATUS,Z
				goto	GetNewLine			;if not then wait for next byte

				clrf	Checksum			;start with checksum zero	

				call	GetHexByte			;get number of program data bytes in line
				andlw	0x1F				;limit number in case of error in file
				movwf	NumWords
				bcf		STATUS,C
				rrf		NumWords,F			;divide by 2 to get number of words

				call	GetHexByte			;get upper half of program start address
				movwf	AddressH

				call	GetHexByte			;get lower half of program start address
				movwf	AddressL

				bcf		STATUS,C
				rrf		AddressH,F			;divide address by 2 to get word address
				rrf		AddressL,F

				call	GetHexByte			;get record type
				xorlw	0x01
				btfsc	STATUS,Z			;check if end of file record (0x01)
				goto	FileDone			;if end of file then all done

				movf	HexByte,W
				xorlw	0x00
				btfss	STATUS,Z			;check if regular line record (0x00)
				goto	LineDone			;if not then ignore line and send '.'

				movlw	0xe0
				addwf	AddressH,W			;check if address < 0x2000
				btfsc	STATUS,C			;which is ID locations and config bits
				goto	LineDone			;if so then ignore line and send '.'

;-----------------------------------------------------------------------------
;Get data bytes and checksum from line of hex file

				movlw	DataArray
				movwf	FSR					;set pointer to start of array
				movf	NumWords,W
				movwf	Counter				;set counter to number of words

GetData:
				call	GetHexByte			;get low data byte
				movwf	INDF				;save in array
				incf	FSR,F				;point to high byte
	
				call	GetHexByte			;get high data byte
				movwf	INDF				;save in array
				incf	FSR,F				;point to next low byte

				decfsz	Counter,F
				goto	GetData

				call	GetHexByte			;get checksum
				movf	Checksum,W			;check if checksum correct
				btfss	STATUS,Z
				goto	ErrorMessage

;-----------------------------------------------------------------------------
;Get saved data one word at a time to program into flash 

				movlw	DataArray
				movwf	FSR					;point to start of array
				movf	NumWords,W
				movwf	Counter				;set counter to half number of bytes

CheckAddress:
				movlw	high StartOfBoot 	;get high byte of address
				subwf	AddressH,W
				btfss	STATUS,C			;test if less than boot code address 
				goto	LoadAddress			;yes so continue with write
				btfss	STATUS,Z			;test if equal to boot code address 
				goto	ErrorMessage		;no so error in high byte of address
	
				movlw	low StartOfBoot		;get low byte of address
				subwf	AddressL,W
				btfsc	STATUS,C			;test if less than boot code address 
				goto	ErrorMessage		;no so error in address
;-----------------------------------------------------------------------------
;Load address and data and write data into flash

LoadAddress:
				movf	AddressH,W			;get high address
				Bank2						;change from bank0 to bank2
				movwf	EEADRH				;load high address
				bcf		STATUS,RP1			;change from bank2 to bank0
				movf	AddressL,W			;get low address
				BSF		STATUS,RP1			;change from bank0 to bank2
				movwf	EEADR				;load low address

LoadData:
				movf	INDF,W				;get low byte from array
				movwf	EEDATA				;load low byte
				incf	FSR,F				;point to high data byte
				movf	INDF,W				;get high byte from array
				movwf	EEDATH				;load high byte
				incf	FSR,F				;point to next low data byte

				BSF		STATUS,RP0			;change from bank2 to bank3
				movlw	0x84				;enable writes to program flash
				movwf	EECON1

				movlw	0x55				; do timed access writes
				movwf	EECON2
				movlw	0xaa
				movwf	EECON2
				bsf		EECON1,WR			; begin writing to flash

				nop							; processor halts here while writing
				nop

				Bank0						; change from bank3 to bank0
				incfsz	AddressL,F			; increment low address byte
				goto	CheckLineDone		; check for rollover
				incf	AddressH,F			; if so then increment high address byte
				movlw	'b'					; line has been programmed so
				call	SerialTransmit		; transmit progress indicator back

CheckLineDone:
				decfsz	Counter,F			; check if all words have been programmed
				goto	CheckAddress		; if not then go program next word

;-----------------------------------------------------------------------------
;Done programming line of file

LineDone:	
				MOVLW	LCD_CLEAR
				CALL	LCDPUTCMD
				MOVLW	LCD_HOME
				CALL	LCDPUTCMD
				movlw	0x01
				xorwf	Prog_Char,F
				movf	Prog_Char,W
				call 	LCDPUTCHAR			; Send progress character to LCD
				movlw	'.'					; line has been programmed so
				call	SerialTransmit		; transmit progress indicator back
				goto	GetNewLine			; go get next line hex file

;-----------------------------------------------------------------------------
;Done programming file so send success indicator and trap execution until reset

FileDone:	
				MOVLW	LCD_CLEAR
				CALL	LCDPUTCMD
				MOVLW	LCD_HOME
				CALL	LCDPUTCMD
				movlw	'S'
				call 	LCDPUTCHAR			; Send Sucess character to LCD
				movlw	'S'					; programming complete so
				call	SerialTransmit		; transmit success indicator back

TrapFileDone:
				goto	TrapFileDone		; all done so wait for reset

;-----------------------------------------------------------------------------
;Error in hex file so send failure indicator and trap error

ErrorMessage:
				MOVLW	LCD_CLEAR
				CALL	LCDPUTCMD
				MOVLW	LCD_HOME
				CALL	LCDPUTCMD
				movlw	'F'
				call 	LCDPUTCHAR			; Send Fail character to LCD
				movlw	'F'					; error occurred so
				call	SerialTransmit		; transmit failure indicator back
TrapError3:
				goto	TrapError3			; trap error and wait for reset


;-----------------------------------------------------------------------------
;Receive two ascii digits and convert into one hex byte
;This routine returns in bank0

GetHexByte:
				call	SerialReceive		; get new byte from serial port
				addlw	0xbf				; add -'A' to Ascii high byte
				btfss	STATUS,C			; check if positive
				addlw	0x07				; if not, add 17 ('0' to '9')
				addlw	0x0a				; else add 10 ('A' to 'F') 
				movwf	HexByte				; save nibble
				swapf	HexByte,F			; move nibble to high position

				call	SerialReceive		; get new byte from serial port
				addlw	0xbf				; add -'A' to Ascii low byte
				btfss	STATUS,C			; check if positive
				addlw	0x07				; if not, add 17 ('0' to '9')
				addlw	0x0a				; else add 10 ('A' to 'F') 
				iorwf	HexByte,F			; add low nibble to high nibble
				movf	HexByte,W			; put result in W reg
				addwf	Checksum,F			; add to cumulative checksum
				return

;-----------------------------------------------------------------------------
;Set up USART for asynchronous comms
;Routine is only called once and can be placed in-line saving a call and return
;This routine returns in bank0

SerialSetup:
				BSF		STATUS, RP0			; Select Register page 0
				movlw	BAUD_9600			;set baud rate 9600 
				movwf	SPBRG
				bsf		TXSTA,BRGH			;baud rate high speed option
				bcf		TXSTA,SYNC			;set async mode
				bsf		TXSTA,TXEN			;enable transmission
				BCF		STATUS, RP0			; Select Register page 0
				bsf		RCSTA,CREN			;enable reception
				bsf		RCSTA,SPEN			;enable serial port
				return

;-----------------------------------------------------------------------------
;Wait for byte to be received in USART and return with byte in W
;This routine returns in bank0

SerialReceive:
				Bank0						;change from unknown bank to bank0
				btfss	PIR1,RCIF			;check if data received
				goto	$-1					;wait until new data
				movf	RCREG,W				;get received data into W
				return

;-----------------------------------------------------------------------------
;Transmit byte in W register from USART
;This routine returns in bank0

SerialTransmit:
				Bank0						;change from unknown bank to bank0
				btfss	PIR1,TXIF			;check that buffer is empty
				goto	$-1
				movwf	TXREG				;transmit byte
				return

;*****************************************************************************
; LCD Module Subroutines
;*****************************************************************************
;
;=============================================================================
; LCDINIT
; Initilize LC-Display Module
; Should be modified to your needs (i.e. display type, cursor on/off, etc.)
; OK
;=============================================================================
LCDINIT
											; Busy-flag is not yet valid
				CLRF	PORTE				; ALL PORT output should output Low.
											; power-up delay
				MOVLW	.250
				CALL	X_DELAY100			; 250 * 0.1mS = 25mS
				MOVLW	0x1c				; Turn on LCD Driver
				CALL	LCDPUTCMD
				MOVLW	0x14				; Turn on Character display
				CALL	LCDPUTCMD
				MOVLW	0x28				; 2X12 Character display (actually 24X1)
				CALL	LCDPUTCMD
				MOVLW	0x4F				; Highest Contrast
				CALL	LCDPUTCMD
				MOVLW	LCD_CLEAR
				CALL	LCDPUTCMD

				MOVLW	.250
				CALL	X_DELAY100			; 250 * 0.1mS = 25mS

				RETURN

;=============================================================================
; READ_FP
; Read the front panel switches and store state in FP_Switch
; OK
;=============================================================================
FP_READ
				Bank1						; Switch to bank0
				MOVLW	0x0FF				; Set PORTD for input
				MOVWF	TRISD
				BCF		STATUS, RP0			; Select Register page 0

				movf	PORTD,W
				movwf	FP_Switch			; Save Switch State

				BSF		STATUS, RP0			; Select Register page 1
				CLRF	TRISD				; Set PORTD for output
				BCF		STATUS, RP0			; Select Register page 0
				RETURN

;=============================================================================
; LCDBUSY
; Looks to make sure no buttons are pressed before
; writing to the LCD
;=============================================================================
LCDBUSY
				BSF		STATUS,RP0			; Select Register page 1
				MOVLW	0x0FF				; Set PORTD for input
				MOVWF	TRISD
				BCF		STATUS, RP0			; Select Register page 0
SWITCHDOWN
				comf	PORTD,W
				btfss	STATUS,Z
				GOTO 	SWITCHDOWN			; Wait until no buttons are pushed

				BSF		STATUS, RP0			; Select Register page 1
				CLRF	TRISD				; Set PORTD for output
				BCF		STATUS, RP0			; Select Register page 0

				RETURN

;=============================================================================
; LCDSDDA
; Sets the Display-Data-RAM address. DDRAM data is read/written after
;  this setting.
; Required DDRAM address must be set in W
;  b0-6	: required DDRAM address
;  b7	: don't care
; OK
;=============================================================================
LCDSDDA
				IORLW	0x0E0				; Function set
				CALL	LCDPUTCMD
				RETURN

;=============================================================================
; LCDPUTCHAR
; Sends character to LCD
; Required character must be in W
; OK
;=============================================================================
LCDPUTCHAR
				MOVWF	LCD_TEMP			; Character to be sent is in W
				CALL	LCDBUSY				; Wait for LCD to be ready
				movlw	0x04				; Set LCD in read mode
				movwf	PORTE				; Set LCD in data mode
				CALL	DELAY001			; 1 usec
				movlw	0x05
				movwf	PORTE				; LCD E-line High
				MOVF	LCD_TEMP, W
				NOP
				MOVWF	PORTD				; Send data to LCD
				CALL	DELAY001
				movlw	0x04				; 
				movwf	PORTE				; 
				clrf	PORTE				; LCD E-line Low
				MOVLW	.5
				CALL	X_DELAY100			; 5 * 0.1mS = 0.5mS
				RETURN

;=============================================================================
; LCDPUTCMD
; Sends command to LCD
; Required command must be in W
; OK
;=============================================================================
LCDPUTCMD
				MOVWF	LCD_TEMP			; Command to be sent is in W
				CALL	LCDBUSY				; Wait for LCD to be ready
				movlw	0x01
				movwf	PORTE				; LCD E-line High
				CALL	DELAY001			; 1 usec
				MOVF	LCD_TEMP, W
				NOP
				MOVWF	PORTD				; Send data to LCD
				CALL	DELAY001			; 1 usec
				clrf	PORTE				; LCD E-line Low
				MOVLW	.5
				CALL	X_DELAY100			; 5 * 0.1mS = 0.5mS
				RETURN

;*****************************************************************************
; Delay_time	= ((DELAY_value * 3) + 4) * Cycle_time
; DELAY_value	= (Delay_time - (4 * Cycle_time)) / (3 * Cycle_time)
;
; i.e. (@ 4MHz crystal)
; Delay_time	= ((32 * 3) + 4) * .2uSec
;		= 20uSec
; DELAY_value	= (100uSec - 0.8) / 0.6
;		= 165.33
;		= 165
;*****************************************************************************
DELAY100		MOVLW	.165				; +1		1 cycle
				MOVWF	DELAY				; +2		1 cycle
DELAY100_LOOP	DECFSZ	DELAY, F			; step 1	1 cycle
				GOTO	DELAY100_LOOP		; step 2	2 cycles
DELAY100_END	RETURN						; +3		2 cycles
;
;
X_DELAY100		MOVWF	X_DELAY				; +1		1 cycle
X_DELAY100_LOOP	CALL	DELAY100			; step1		wait 100uSec
				DECFSZ	X_DELAY, F			; step2		1 cycle
				GOTO	X_DELAY100_LOOP		; step3		2 cycles
X_DELAY100_END	RETURN						; +2		2 cycles
;
; About a 1 usec delay at 20 MHz clock
DELAY001		NOP
				NOP
				NOP
				RETURN


		END

